/*****************************************************************************
 * yahoofxfer.c
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * Copyright (C) 2006 Stefan Sikora
 *****************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>

#include <gtk/gtk.h>

#include "config.h"

#include "gyach.h"
#include "main.h"
#include "yahoochat.h"
#include "util.h"
#include "yahoofxfer.h"
#include "sounds.h"
#include "friends.h"
#include "profname.h"
#include "conference.h"
#include "users.h"
#include "pmwindow.h"
#include "packet_handler.h"
#include "interface.h"

#include "gy_config.h"
#include "theme_support.h"
#include "gyachi_lib.h"
#include "gytreeview.h"

GList *file_xfer_list = NULL;
GList *relay_server_blacklist = NULL;

/* not a correct urlencode, just to please yahoo's server */
char *url_encode(char *str)
{
	char *buf;
	char *ptr;
	int   i;

	if (str == NULL) {
		return(strdup(""));
	}

	/* We need to allocate a return string.
	 * It needs to contain the original string, with all the ^b converted to "%02"
	 */

	/* count the number of occurances of a ^b */
	for (ptr=str, i=0; (ptr=strchr(ptr, 0x02)); i++, ptr++);

	if (i == 0) {
		return (strdup(str));
	}

	buf = malloc(strlen(str) + (i*2) +1);

	ptr = buf;
	while (*str) {
		if (*str != 0x02) {
			*ptr++ = *str++;
		}
		else {
			*ptr++ = '%';
			*ptr++ = '0';
			*ptr++ = '2';
			str++;
		}
	}
	*ptr = 0;

	return buf;
}

struct fxfile *create_fxfile_info(char *who, char *filename, char *filekey, int file_length)
{
	struct fxfile *file_info;

	file_info=malloc(sizeof(struct fxfile));
	file_info->who=strdup(who);
	file_info->filekey= filekey  ? strdup(filekey)  : NULL;
	file_info->filename=filename ? strdup(filename) : NULL;
	file_info->file_length=file_length;
	file_info->fullfilename=NULL;
	file_info->hostname=NULL;
	file_info->server=NULL;
	file_info->token=NULL;
	file_info->dl_tree=NULL;
	file_info->start_time=0;
	file_info->show_meter=auto_open_dl_manager;
	file_info->cancel_transfer=0;
	file_info->status=0;
	file_info->open_callback=NULL;
	file_info->open_arg=NULL;
	file_info->finish_callback=NULL;
	file_info->finish_arg=NULL;
	file_xfer_list = g_list_append(file_xfer_list, file_info);
	memset(&file_info->dl_manager_iter, 0, sizeof(file_info->dl_manager_iter));
	return(file_info);
}

void free_fxfile_info(struct fxfile *file_info) {
	if (file_xfer_list) {
		if (g_list_index(file_xfer_list, file_info) < 0) {
			/* the file_info NOT FOUND in the file_xfer_list
			 * Generally, this should not happen. The case
			 * where it might happen, is if we're called from
			 * the update_meter() callback, and the file_info
			 * has already been removed/freed...
			 */
			if (debug_packets) {
				fprintf(packetdump_fp, "free_fxfile_info: ignoring attempt to free a freed file_info\n");
			}
			return;
		}
	}
	else {
		/* the file_info list was empty to begin with. This, then, is a 
		 * call to free an already freed file_info item...
		 * Ignore.
		 */
		if (debug_packets) {
			fprintf(packetdump_fp, "free_fxfile_info: ignoring attempt to free a freed file_info\n");
		}
		return;
	}
	file_xfer_list = g_list_remove(file_xfer_list, file_info);

	if (debug_packets) {
		fprintf(packetdump_fp, "free_fxfile_info: file_xfer_list is %d deep\n", g_list_length(file_xfer_list));
	}

	if (file_info->finish_callback) {file_info->finish_callback(file_info->finish_arg);}
	if (file_info->fullfilename)    {free(file_info->fullfilename); file_info->fullfilename=NULL;}
	if (file_info->filename)        {free(file_info->filename);     file_info->filename=NULL;}
	if (file_info->filekey)         {free(file_info->filekey);      file_info->filekey=NULL;}
	if (file_info->hostname)        {free(file_info->hostname);     file_info->hostname=NULL;}
	if (file_info->server)          {free(file_info->server);       file_info->server=NULL;}
	if (file_info->who)             {free(file_info->who);          file_info->who=NULL;}
	if (file_info->token)           {free(file_info->token);        file_info->token=NULL;}
	if (file_info->open_arg)        {free(file_info->open_arg);     file_info->open_arg=NULL;}
	free(file_info);
}


gboolean yahoo_relay_server_is_blacklisted(char *server) {
	GList *l_listp;

	for (l_listp=relay_server_blacklist; l_listp; l_listp=g_list_next(l_listp)) {
		if (!strcmp(server, (char *)(l_listp->data))) {
			return 1;
		}
	}

	return 0;
}

struct fxfile *yahoo_find_file_info(char *who, char *server, char *filename, char *filekey) {
	struct fxfile *file_info;
	GList *l_listp;

	if (debug_packets) {
		fprintf(packetdump_fp, "yahoo_find_file_info: file_xfer_list is %d deep\n", g_list_length(file_xfer_list));
	}

	file_info = 0;
	for (l_listp=file_xfer_list; l_listp; l_listp=g_list_next(l_listp)) {
		file_info=(struct fxfile *)(l_listp->data);
		if (strcmp(filekey,  file_info->filekey))  continue;
		if (strcmp(who,      file_info->who))      continue;
		return file_info;
	}

	/* hmmm. this should'nt happen.
	 * If here, then we didn't find our file_info!
	 * Well... log it!
	 */

	if (debug_packets) {
		fprintf(packetdump_fp, "Could not find file_info for who:'%s', filename: '%s', filekey: '%s'\n", 
			who,
			filename,
			filekey);
	}

	if (capture_fp) {
		fprintf(capture_fp, "Could not find file_info for who:'%s', filename: '%s', filekey: '%s'\n", 
			who,
			filename,
			filekey);
	}

	return 0;
}

struct fxfile *yahoo_find_file_info_by_iter(GtkTreeIter *iter) {
	struct fxfile *file_info;
	GList *l_listp;

	if (debug_packets) {
		fprintf(packetdump_fp, "yahoo_find_file_info: file_xfer_list is %d deep\n", g_list_length(file_xfer_list));
	}

	file_info = 0;
	for (l_listp=file_xfer_list; l_listp; l_listp=g_list_next(l_listp)) {
		file_info=(struct fxfile *)(l_listp->data);
		if ((file_info->dl_manager_iter.stamp != iter->stamp) ||
		    (file_info->dl_manager_iter.user_data != iter->user_data)) {
			continue;
		}
		return file_info;
	}

	/* hmmm. this should'nt happen.
	 * If here, then we didn't find our file_info!
	 * Well... log it!
	 */

	if (debug_packets) {
		fprintf(packetdump_fp, "Could not find file_info by iter\n");
	}

	if (capture_fp) {
		fprintf(capture_fp, "Could not find file_info by iter\n");
	}

	return 0;
}


gboolean is_valid_file_info(struct fxfile *file_info) {
	GList *l_listp;

	if (debug_packets) {
		fprintf(packetdump_fp, "is_valid_file_info: file_xfer_list is %d deep\n", g_list_length(file_xfer_list));
	}

	for (l_listp=file_xfer_list; l_listp; l_listp=g_list_next(l_listp)) {
		if (l_listp->data == file_info) {
			return 1;
		}
	}

	return 0;
}

/*
 * -----------------------------------------------------------------------------------------
 */

void on_dl_close_clicked(GtkButton *button, gpointer user_data)
{
	GtkWidget *dlmanager = (GtkWidget *)user_data;

	gtk_widget_hide(dlmanager);
	g_object_set_data(G_OBJECT(dlmanager), "hide", dlmanager);
}

void on_remove_dl_entry(GtkWidget *widget, gpointer user_data)
{
	struct fxfile *file_info = user_data;
	GtkTreeModel *model;
	GtkTreeIter  *iter;

	model = g_object_get_data(G_OBJECT(widget), "model");
	iter  = g_object_get_data(G_OBJECT(widget), "iter");

	gtk_list_store_remove(GTK_LIST_STORE(model), iter);
	if (file_info && is_valid_file_info(file_info)) {
		memset(&file_info->dl_manager_iter, 0, sizeof(file_info->dl_manager_iter));
	}
}

void on_blacklist_dl_entry(GtkWidget *widget, gpointer user_data)
{
	GtkTreeModel *model;
	GtkTreeIter  *iter;
	gchar        *host_addr;

	model = g_object_get_data(G_OBJECT(widget), "model");
	iter  = g_object_get_data(G_OBJECT(widget), "iter");

	host_addr = NULL;
	gtk_tree_model_get(model, iter, GYFILEDL_RELAY_SERVER, &host_addr, -1);

	if (!yahoo_relay_server_is_blacklisted(host_addr)) {
		relay_server_blacklist=g_list_prepend(relay_server_blacklist, host_addr);
	}
	else {
		free(host_addr);
	}
}

void on_cancel_transfer(GtkWidget *widget, gpointer user_data)
{
	struct fxfile *file_info = user_data;

	if (file_info && is_valid_file_info(file_info) && file_info->status == INPROGRESS) {
		file_info->cancel_transfer = 1;
		ymsg_fxfer_cancel(ymsg_sess, file_info->who, file_info->filekey);
	}
}

void on_destroy_remove_cleanup(GtkWidget *widget, gpointer user_data)
{
	free(user_data);
}

void on_destroy_vbox_cleanup(GtkWidget *widget, gpointer user_data)
{
	struct fxfile *file_info;
	GList *l_listp;

	file_info = 0;
	for (l_listp=file_xfer_list; l_listp; l_listp=g_list_next(l_listp)) {
		file_info=(struct fxfile *)(l_listp->data);
		if (file_info->dl_tree == user_data) {
			file_info->dl_tree = NULL;
		}
	}

}


gboolean on_dl_file_button_press(GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
	struct fxfile *file_info;
	GtkTreePath *path;
	GtkTreeViewColumn *column;
	GtkTreeModel *model;
	GtkTreeIter   iter;
	int rv;
	static GtkWidget *popup_menu;
	GtkWidget *cancel_button;
	GtkWidget *remove_button;
	GtkWidget *blacklist_button;
	GtkWidget *p_iter;
	int        direction;

	/* Right mouse button clicks: Popup menu */
	if (event->button != 3 || event->type != GDK_BUTTON_PRESS) {
		return FALSE;
	}

	rv = gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(widget), event->x, event->y, &path, &column, NULL, NULL);
	if (!rv) {
		return FALSE;
	}

	model = gtk_tree_view_get_model(GTK_TREE_VIEW(widget));
	memset(&iter, 0, sizeof(iter));
	rv = gtk_tree_model_get_iter(model, &iter, path);
	if (!rv) {
		return FALSE;
	}

	p_iter=malloc(sizeof(GtkTreeIter));
	memcpy(p_iter, &iter, sizeof(GtkTreeIter));

	if (popup_menu ) {
		gtk_widget_destroy(popup_menu);
	}

	popup_menu = gtk_menu_new();

	gtk_tree_model_get(model, &iter, GYFILEDL_DIRECTION, &direction, -1);

	file_info = yahoo_find_file_info_by_iter(&iter);
	if (file_info && (file_info->status == INPROGRESS)) {
		cancel_button = gtk_image_menu_item_new_with_label(_("Cancel Transfer"));
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cancel_button), 
					      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_CANCEL,
										  GTK_ICON_SIZE_MENU)));
		gtk_container_add(GTK_CONTAINER(popup_menu), cancel_button);

		g_signal_connect(G_OBJECT(cancel_button), "activate",
				 G_CALLBACK(on_cancel_transfer),
				 file_info);
	}

	remove_button = gtk_image_menu_item_new_with_label(_("Remove Entry"));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(remove_button), 
				      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_REMOVE,
									  GTK_ICON_SIZE_MENU)));
	gtk_container_add(GTK_CONTAINER(popup_menu), remove_button);

	g_signal_connect(G_OBJECT(remove_button), "activate",
			 G_CALLBACK(on_remove_dl_entry),
			 file_info);

	g_object_set_data(G_OBJECT(remove_button), "model", model);
	g_object_set_data(G_OBJECT(remove_button), "iter",  p_iter);

	if (direction == Sending) {
		blacklist_button = gtk_image_menu_item_new_with_label(_("Blacklist this Relay Server"));
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(remove_button), 
					      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_DELETE,
										  GTK_ICON_SIZE_MENU)));
		gtk_container_add(GTK_CONTAINER(popup_menu), blacklist_button);

		g_signal_connect(G_OBJECT(blacklist_button), "activate",
				 G_CALLBACK(on_blacklist_dl_entry),
				 file_info);

		g_object_set_data(G_OBJECT(blacklist_button), "model", model);
		g_object_set_data(G_OBJECT(blacklist_button), "iter",  p_iter);
	}

	gtk_widget_show_all(popup_menu);

	g_signal_connect(remove_button, "destroy",
			 G_CALLBACK(on_destroy_remove_cleanup),
			 p_iter);

	gtk_menu_popup((GtkMenu *) popup_menu, NULL, NULL, NULL, NULL, 1, 0);

	return FALSE;
}

GtkWidget *create_dl_manager(GtkWidget *pm_session) {
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *my_label;
	GtkWidget *close_button;
	GtkWidget *scrolledwindow;
	GtkWidget *dl_tree;
	char *col_headers[]={"","","","","","","","",NULL};

	/* see create_buddy_list_tab() in interface.c */

	/* 
	   vbox
	   hbox = label "GyachI download Manager"  :  [x]
	   treeview

	   [-] Progress | Filename | Size | Remaining
	*/

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox),  0);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox),  0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	my_label = gtk_label_new("GyachI Download Manager");
	gtk_box_pack_start(GTK_BOX(hbox), my_label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(""), TRUE, TRUE, 0);
	close_button=get_stocki_button(GTK_STOCK_CLOSE,_("Close Download Manager"), GTK_ICON_SIZE_MENU);
	gtk_box_pack_start(GTK_BOX(hbox), close_button, FALSE, FALSE, 0);

	scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
	gtk_box_pack_start(GTK_BOX(vbox), scrolledwindow, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

	col_headers[0]=_("");
	col_headers[1]=_("Progress");
	col_headers[2]=_("Relay Server");
	col_headers[3]=_("Filename");
	col_headers[4]=_("Size");
	col_headers[5]=_("Status");
	col_headers[6]=_("Speed");
	col_headers[7]=_("ETA");

	dl_tree=GTK_WIDGET(create_gy_treeview(GYTV_TYPE_LIST, GYLIST_TYPE_FILEDL, 1, 0, col_headers));
	gtk_container_add(GTK_CONTAINER(scrolledwindow), dl_tree);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledwindow), GTK_SHADOW_NONE);

	g_object_set_data(G_OBJECT(pm_session), "dl_tree", dl_tree);
	g_object_set_data(G_OBJECT(pm_session), "dlmanager", vbox);

	g_signal_connect(close_button, "clicked",
			 G_CALLBACK(on_dl_close_clicked),
			 vbox);
	g_signal_connect(dl_tree, "button-press-event",
			 G_CALLBACK(on_dl_file_button_press),
			 vbox);
	g_signal_connect(vbox, "destroy",
			 G_CALLBACK(on_destroy_vbox_cleanup),
			 dl_tree);

	gtk_paned_pack1(GTK_PANED(pm_session), vbox, TRUE, TRUE);
	gtk_widget_show_all(vbox);

	return(vbox);
}

/*
 * -----------------------------------------------------------------------------------------
 */


int update_meter(gpointer data)
{
	GtkTreeModel *model;
	struct fxfile *file_info = data;
	double  percent;
	int     p_cent;
	char    buff[128];
	char    speed[128];
	char    eta_buf[128];
	GtkTreeView *dl_tree = NULL;
	GtkTreeIter iter;
	GdkPixbuf *imbuf;
	time_t     now=time(NULL);
	double     bps;
	time_t     eta;
	int        eta_min,     eta_sec;
	int        elapsed_min, elapsed_sec;
	int        total_min,   total_sec;

	dl_tree = file_info->dl_tree;
	if (dl_tree) {
		imbuf=NULL;
		model=gtk_tree_view_get_model(dl_tree);

		memset(&iter, 0, sizeof(iter));
		if (memcmp(&iter, &file_info->dl_manager_iter, sizeof(iter))) {
			/* handle the case where the d/l line has been removed */
			if (!gtk_list_store_iter_is_valid(GTK_LIST_STORE(model), &file_info->dl_manager_iter)) {
				memset(&file_info->dl_manager_iter, 0, sizeof(iter));
			}
		}
		if (now == file_info->start_time) now++; /* avoid division by zero */
		if (memcmp(&iter, &file_info->dl_manager_iter, sizeof(iter))) {
			percent=((float)file_info->bytes_xfer/(float)file_info->file_length)*100.0;
			p_cent = percent +0.5;
			bps = 1.0*file_info->bytes_xfer/(now-file_info->start_time);
			eta = (now-file_info->start_time)*100/percent + file_info->start_time-now;

			eta_min = eta/60;
			eta_sec = eta%60;

			elapsed_min = (now-file_info->start_time)/60;
			elapsed_sec = (now-file_info->start_time)%60;

			total_min = (now-file_info->start_time+eta)/60;
			total_sec = (now-file_info->start_time+eta)%60;

			sprintf(buff, "%d%% (%d of %d bytes)", p_cent, file_info->bytes_xfer, file_info->file_length);
			sprintf(speed, "%.2f %s", (bps > (1024*1024)) ? bps/(1024*1024) : bps/1024, (bps > (1024*1024)) ? "MiB" : "KiB");
			sprintf(eta_buf, "Elapsed: %d:%02d, Remaining: %d:%02d, Total: %d:%02d",
				elapsed_min, elapsed_sec,
				eta_min, eta_sec,
				total_min, total_sec);

			gtk_list_store_set(GTK_LIST_STORE(model), &file_info->dl_manager_iter, 
					   GYFILEDL_PROGRESS, p_cent,
					   GYFILEDL_STATUS,   buff,
					   GYFILEDL_SPEED,    speed,
					   GYFILEDL_ETA,      eta_buf,
					   -1);
			switch (file_info->status) {
			case SUCCESS:
				imbuf= get_pixbuf_from_stock_id(chat_window, GTK_STOCK_APPLY, GTK_ICON_SIZE_MENU);
				break;

			case CANCELLED:
				imbuf= get_pixbuf_from_stock_id(chat_window, GTK_STOCK_CANCEL, GTK_ICON_SIZE_MENU);
				break;

			case UNKNOWN:
				imbuf= get_pixbuf_from_stock_id(chat_window, GTK_STOCK_STOP, GTK_ICON_SIZE_MENU);
				break;

			default:
				break;
			}
		}

		if (imbuf) {
			gtk_list_store_set(GTK_LIST_STORE(model), &file_info->dl_manager_iter, 
					   GYFILEDL_PIX, imbuf,
					   -1);
			g_object_unref(imbuf);
		}
	}

	if ((file_info->status == SUCCESS)   ||
	    (file_info->status == CANCELLED) ||
	    (file_info->status == UNKNOWN))   {
		free_fxfile_info(file_info);
		return(FALSE);
	}


	return TRUE;
}

void create_meter(PM_SESSION *pm_sess, struct fxfile *file_info, DIRECTION direction, int show) {
	GtkWidget *pm_session = NULL;
	GtkWidget *dlmanager = NULL;
	GtkTreeView *dl_tree = NULL;
	GtkTreePath *path;

	if (pm_sess) {
		pm_session = pm_sess->pm_window;
	}
	if (pm_session) {
		dlmanager = g_object_get_data(G_OBJECT(pm_session), "dlmanager");
	}

	if (!dlmanager) {
		dlmanager = create_dl_manager(pm_session);
	}

	dl_tree = g_object_get_data(G_OBJECT(pm_session), "dl_tree");

	if (dl_tree) {
		GtkTreeModel *model;
		GtkTreeIter dl_manager_iter;
		char size[10];
		GdkPixbuf *imbuf=NULL;

		sprintf(size, "%d %s",
			file_info->file_length/(1024*1024) > 0 ? file_info->file_length/(1024*1024) : file_info->file_length/1024,
			file_info->file_length/(1024*1024) > 0 ? "Mb" : "Kb");

		model=gtk_tree_view_get_model(dl_tree);
		gtk_list_store_append(GTK_LIST_STORE(model), &dl_manager_iter);

		file_info->dl_tree         = dl_tree;
		file_info->dl_manager_iter = dl_manager_iter;
		file_info->start_time      = time(NULL);
		imbuf= get_pixbuf_from_stock_id(chat_window, direction == Sending ? GTK_STOCK_GO_UP : GTK_STOCK_GO_DOWN, GTK_ICON_SIZE_MENU);
		gtk_list_store_set(GTK_LIST_STORE(model), &file_info->dl_manager_iter, 
				   GYFILEDL_PIX,          imbuf,
				   GYFILEDL_PROGRESS,     0,
				   GYFILEDL_RELAY_SERVER, file_info->server,
				   GYFILEDL_FILENAME,     file_info->filename,
				   GYFILEDL_SIZE,         size,
				   GYFILEDL_STATUS,       "Not Started",
				   GYFILEDL_SPEED,        "N/A",
				   GYFILEDL_ETA,          "N/A",
				   GYFILEDL_DIRECTION,    direction,
				   -1);
		
		if (imbuf) {
			g_object_unref(imbuf);
		}

		if (show) {
			gtk_widget_show(dlmanager);
		}
		else {
			gtk_widget_hide(dlmanager);
		}

		/* scroll the new line onto the screen */
		path = gtk_tree_model_get_path(model, &file_info->dl_manager_iter);
		gtk_tree_view_scroll_to_cell(dl_tree, path, NULL, TRUE, 1.0, 0.0);
		gtk_tree_path_free(path);

		/* update the meter every half sec (500 ms) */
		g_timeout_add(500, update_meter, file_info);
	}
}


/*
 * -----------------------------------------------------------------------------------------
 */

/*
 * yahoo_fxfer_offer_msg dialog support
 *
 * [ok] button clicked
 *
 */
void on_fxfer_accept(GtkWidget *button, gpointer user_data) {
	GtkWidget *tmp_widget;
	struct fxfile *file_info;

	/* get embedded data */
	file_info = g_object_get_data(G_OBJECT(button), "file_info");
	if (file_info) {
		if (file_info->cancel_transfer) {
			/* Yes, this can happen.
			 * remote user cancelled before we confirmed!
			 */
			free_fxfile_info(file_info);
		}
		else {
			ymsg_fxfer_accept(ymsg_sess, file_info->who, file_info->filekey);
		}
	}
	tmp_widget = g_object_get_data(G_OBJECT(button), "mywindow");
	if (tmp_widget) gtk_widget_destroy(tmp_widget);
}

/*
 * yahoo_fxfer_offer_msg dialog support
 *
 * [cancel] button clicked
 *
 */
void on_fxfer_reject(GtkWidget *button, gpointer user_data) {
	GtkWidget *tmp_widget;
	struct fxfile *file_info;

	/* get embedded data */
	file_info = g_object_get_data(G_OBJECT(button), "file_info");
	if (file_info) {
		ymsg_fxfer_reject(ymsg_sess, file_info->who, file_info->filekey);
		free_fxfile_info(file_info);
	}
	tmp_widget = g_object_get_data(G_OBJECT(button), "mywindow");
	if (tmp_widget) gtk_widget_destroy(tmp_widget);
}

/*
 * yahoo_fxfer_offer_msg dialog support
 *
 * call back to destroy window when [x] kill window icon is clicked!
 * Almost the same as [cancel], except that the callback is slightly
 * different, and expects a true/false return.
 * False to propogate the event further.
 */
gboolean on_fxfer_destroy(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	on_fxfer_reject(user_data, NULL);
	return(FALSE);
}

/* handles incoming file-offers */
void yahoo_fxfer_offer_msg(char *who, char *filename, char *filekey, int filelength)
{
	char buff[512];
	GtkWidget *okbutton=NULL;
	GtkWidget *cbutton=NULL;
	int accept_pm;
	struct fxfile *file_info;
	PM_SESSION *pm_sess;
	GtkWidget *parent;
	GtkWidget *window;

	accept_pm = get_pm_perms(who);
	if ( ! accept_pm) {					
		send_rejection_message(0);
		return;
	}

	if (allow_no_sent_files) {
		send_rejection_message(0);
		return;
	}

	/* get parent window */
	if ( !(pm_sess = find_pm_session(who)) ) {
		/* open a new window if one doesn't exist already */
	        pm_sess=new_pm_session(who);
	}
	parent = pm_sess->pm_notebook->window;
	if ( auto_raise_pm ) {
		focus_pm_entry(pm_sess);
	}

	snprintf(buff, sizeof(buff)-1, _("The Yahoo user <b>%s</b> is sending you the file:\n<tt>%s</tt> (%d %s)\n\nDo you want to accept?"),
		 who,
		 filename,
		 filelength/(1024*1024) > 0 ? filelength/(1024*1024) : filelength/1024,
		 filelength/(1024*1024) > 0 ? "Mb" : "Kb");

	okbutton = show_confirm_dialog_config_p(parent, buff, _("Yes"), _("No"), 0);
	if (!okbutton) {
		ymsg_fxfer_reject(ymsg_sess, who, filekey); 
		return;
	}
	gtk_widget_grab_focus(okbutton);

	file_info=create_fxfile_info(who, filename, filekey, filelength);

	g_object_set_data(G_OBJECT(okbutton), "file_info", file_info);
	g_signal_handlers_block_by_func(G_OBJECT(okbutton), on_close_ok_dialog, NULL);
	g_signal_connect(G_OBJECT(okbutton), "clicked", G_CALLBACK(on_fxfer_accept), NULL);

	cbutton = g_object_get_data(G_OBJECT(okbutton), "cancel");
	if (cbutton) {
		g_signal_connect(G_OBJECT(cbutton), "clicked", G_CALLBACK(on_fxfer_reject), NULL);
		g_object_set_data(G_OBJECT(cbutton), "file_info", file_info);
	}

	window = g_object_get_data(G_OBJECT(okbutton), "mywindow");
	g_signal_handlers_block_by_func(G_OBJECT(window), on_close_ok_dialogw, NULL);
	g_signal_connect(G_OBJECT(window), "delete_event", G_CALLBACK(on_fxfer_destroy), okbutton);

	play_sound_event(SOUND_EVENT_OTHER);
}

/*
 * -----------------------------------------------------------------------------------------
 */

/* writes information about filetransfer to chatwindow+pm-window */
void yahoo_fxfer_filemessage(char *who, char *filename, char *msg){
	char buf[2048];

	if (msg) {
		snprintf(buf, 256, "  %s%s** ",YAHOO_STYLE_BOLDON, YAHOO_COLOR_PURPLE);
		strncat(buf, msg, sizeof(buf)-strlen(buf)-1-strlen(YAHOO_STYLE_BOLDOFF));
		strcat(buf, YAHOO_STYLE_BOLDOFF);
		strcat(buf, "\n");
		buf[sizeof(buf)-1] = 0;
	}
	else {
		snprintf(buf, 256, "  %s%s** %s: ",YAHOO_STYLE_BOLDON, YAHOO_COLOR_PURPLE, _("User"));
		strcat(buf, "'");
		strncat(buf, who, 80);
		strcat(buf, "' ( ");
		strncat(buf, (char *)get_screenname_alias(who), 80);
		strcat(buf," ) ");
		strncat(buf, _("sent a file that you saved as: "), 90);

		strncat(buf, _utf(filename), 1024); // avoid buffer overflow
		strcat(buf, " **\n");

		strcat(buf, YAHOO_STYLE_BOLDOFF);
		strcat(buf, "\n");
	}
	if (enter_leave_timestamp) {
		append_timestamp(chat_window, NULL);
	}
	append_char_icon_text(NULL, GYACHI_PM_FILE, GTK_ICON_SIZE_BUTTON);
	append_to_textbox(chat_window, NULL, buf);
	append_to_open_pms(who, buf, GYACHI_PM_FILE, 1);
	
	play_sound_event(SOUND_EVENT_OTHER);
} /* yahoo_fxfer_filemessage */


/* for now just notify about p2p-file waiting */
void yahoo_fxfer_getfile_p2p(char *who, char *url, char *filename) {
	yahoo_fxfer_filemessage(who, filename, url);
}

/*
 * -----------------------------------------------------------------------------------------
 */

void *yahoo_fxfer_getfile_thread(void *arg) {
	int sock = -1;
	struct sockaddr_in sa;
	int readct;
	int bytes_read;
	char buffer[513];
	char status_msg[512];
	char *url;
	char *encoded;
	char *fetchurl;
	char *fetchurl_strings[] = {
		"/relay?token=",	// 0
		0,			// 1 - encoded
		"&sender=",		// 2
		0,			// 3 - file_info->who
		"&recver=",		// 4
		0,			// 5 - select_profile_name(YMSG_NOTIFY, 0))
		0			// 6
	};
	char *HEAD_strings[] = {
		"HEAD ",		// 0
		0,			// 1 - fetchurl
		" HTTP/1.1\r\n"
		"Accept: */*\r\n"
		"Cookie: ",		// 2
		0,			// 3 - ymsg_sess->cookie
		"\r\n"
		"User-Agent: " GYACH_USER_AGENT "\r\n"
		"Host: ",		// 4
		0,			// 5 - file_info->server
		"\r\n"
		"Content-Length: 0\r\n"
		"Cache-Control: no-cache\r\n"
		"\r\n",			// 6
		0
	};
	char *GET_strings[] = {
		"GET ",			// 0
		0,			// 1 - fetchurl
		" HTTP/1.1\r\n"
		"Cookie: ",		// 2
		0,			// 3 - ymsg_sess->cookie
		"\r\n"
		"User-Agent: " GYACH_USER_AGENT "\r\n"
		"Host: ",		// 4
		0,			// 5 - file_info->server
		"\r\n"
		"Connection: Keep-Alive\r\n"
		"\r\n",			// 6
		0
	};
	FILE *f;
	int err;
	char *ptr;
	int  count;
	int  file_length;
	struct fxfile *file_info = arg;
	GtkWidget *parent; /* either pm of "who", or main chat window */

	file_info->bytes_xfer=0;
	/* get parent window */
	gdk_threads_enter();
	// * FIXME * get parent before thread starts.
	// * FIXME * ? maybe put it in the fxfile struct ?
	parent = find_pms_window(file_info->who);
	if (!parent) parent=chat_window;
	gtk_window_present(GTK_WINDOW(parent));
	gdk_flush();

	// ack of token
	/* NOTE: the following ymsg_ call *MUST* be done inside a gdk_threads enter
	 *       so as to synchronize access to all the ymsg_ shared buffers.
	 *       the ymsg_* routines just are NOT thread_safe... *sigh*
	 */
	if (!file_info->finish_callback) {
		ymsg_fxfer_relayaccept(ymsg_sess, file_info->who, file_info->filename, file_info->filekey, file_info->token);
	}
	gdk_threads_leave();

	memset(&sa, 0, sizeof(sa));
	sa.sin_addr.s_addr = inet_addr(file_info->server);
	sa.sin_family = AF_INET;
	sa.sin_port = htons(80);
	
	if ((sock=socket(AF_INET, SOCK_STREAM, 6)) == -1) {
		gdk_threads_enter();
		show_ok_dialog_p(parent, _("Socket error!"));
		free_fxfile_info(file_info);
		gdk_flush();
		gdk_threads_leave();
		return(0);
	}

	if (connect(sock, (struct sockaddr*)&sa, sizeof(sa)) == -1) {
		gdk_threads_enter();
		show_ok_dialog_p(parent, _("Could not connect to relay server"));
		free_fxfile_info(file_info);
		gdk_flush();
		gdk_threads_leave();
		close(sock);
		return(0);
	}

	encoded = url_encode(file_info->token);
	fetchurl_strings[1] = encoded;
	fetchurl_strings[3] = file_info->who;
	fetchurl_strings[5] = select_profile_name(YMSG_NOTIFY, 0);
	fetchurl=build_string(fetchurl_strings);
	free(encoded);
	encoded=NULL;

	// send the "HEAD-command"
	HEAD_strings[1] = fetchurl;
	HEAD_strings[3] = ymsg_sess->cookie;
	HEAD_strings[5] = file_info->server;
	url = build_string(HEAD_strings);

	write(sock, url, strlen(url));
	free(url);
	url=NULL;

	// now receive the answer...
	bytes_read = read(sock, (char *)&buffer, (sizeof(buffer)-1));
	close(sock);
	
	if (debug_packets) {
		int i;
		fprintf(packetdump_fp, "received from relay server: size: %d\n", bytes_read);
		for (i=0; i<bytes_read; i++) {
			fputc(buffer[i], packetdump_fp);
		}
		fprintf(packetdump_fp, "\n");
	}

	// new socket for the GET	
	memset(&sa, 0, sizeof(sa));
	sa.sin_addr.s_addr = inet_addr(file_info->server);
	sa.sin_family = AF_INET;
	sa.sin_port = htons(80);
	
	if ((sock=socket(AF_INET, SOCK_STREAM, 6)) == -1) {
		gdk_threads_enter();
		show_ok_dialog_p(parent, _("Socket error!"));
		free_fxfile_info(file_info);
		gdk_flush();
		gdk_threads_leave();
		free(fetchurl);
		fetchurl=NULL;
		return(0);
	}

	if (connect(sock, (struct sockaddr*)&sa, sizeof(sa)) == -1) {
		gdk_threads_enter();
		show_ok_dialog_p(parent, _("Could not connect to relay server"));
		free_fxfile_info(file_info);
		gdk_flush();
		gdk_threads_leave();
		close(sock);
		free(fetchurl);
		fetchurl=NULL;
		return(0);
	}
	memset(&buffer, 0, sizeof(buffer));

	// send the "GET-command"
	GET_strings[1] = fetchurl;
	GET_strings[3] = ymsg_sess->cookie;
	GET_strings[5] = file_info->server;
	url = build_string(GET_strings);
	free(fetchurl);
	fetchurl=NULL;

	write(sock, url, strlen(url));
	free(url);
	url=NULL;

	// get the http-header
	/* The header will look like this:
	 *     HTTP/1.0 200\r\n
	 *     Server: YHttpServer\r\n
	 *     Connection: close\r\n
	 *     Content-Type: text/html\r\n
	 *     Content-Length: 33032\r\n
	 *     \r\n
	 *
	 * The actual content MIGHT start on the next byte, *OR* the received packet
	 * might be at the end of the buffer, and then the data begins at the start
	 * of the next packet. I'm guessing that in the earlier days, this packet
	 * would END with the Content-Length: ###\r\n\r\n, and recently, sometime
	 * 2005/2006, yahoo started to just send the packet header right infront of
	 * the file IN THE SAME PACKET... That's mostly what I'm seeing these days...
	 */

	buffer[sizeof(buffer)-1]=0;
	readct = read(sock, (char *)&buffer, sizeof(buffer)-1);
	if (debug_packets) {
		int i;
		fprintf(packetdump_fp, "received from relay server: size: %d\n", readct);
		for (i=0; i<readct; i++) {
			fputc(buffer[i], packetdump_fp);
		}
		fprintf(packetdump_fp, "\n");
	}

	// now get the file
	if (!(f = fopen(file_info->fullfilename, "wb"))) {
		snprintf(status_msg, 500, "%s:\n%s", _("File could not be opened for output"), file_info->fullfilename);
		gdk_threads_enter();
		show_ok_dialog_p(parent, status_msg);
		free_fxfile_info(file_info);
		gdk_flush();
		gdk_threads_leave();
		close(sock);
		return(0);
	}

	/* parse the header. Look for
	 * Content-Length: -- if we see, then extract the filesize.
	 * \r\n\r\n -- if we see, then ignore up to the \r\n\r\n, and everything
	 * else in the packet is file content.
	 */
	file_info->bytes_xfer = 0;
	ptr=buffer;
	while (ptr && strncmp(ptr, "\r\n\r\n", 4)) {
		if (strncmp(ptr, "\r\n", 2) == 0) ptr += 2;
		if (strncasecmp(ptr, "Content-Length: ", 16) == 0) {
			ptr+=16;
			file_length=atoi(ptr);
			if (file_length != file_info->file_length) {
				if (capture_fp) {
					fprintf(capture_fp, "yahoo file_length (%d) does not match http header file_length (%d)\n",
						file_info->file_length, file_length);
					file_info->file_length=file_length;
				}
			}
		}
		ptr=strstr(ptr, "\r\n");
	}

	/* create the progress meter box */
	gdk_threads_enter();
	create_meter(find_pm_session(file_info->who), file_info, Receiving, file_info->show_meter);
	gdk_flush();
	gdk_threads_leave();
	file_info->status = INPROGRESS;

	/* at here, ptr is either a null pointer, or is sitting on \r\n\r\n */
	if (ptr) {
		count=ptr-buffer+4;
		if (count < readct) {
			fwrite(ptr+4, 1, readct-count, f);
			file_info->bytes_xfer = readct-count;
		}
		if (debug_packets) {
			fprintf(packetdump_fp, "part of file is in initial packet: %d bytes\n", file_info->bytes_xfer);
		}
	}

	while (!file_info->cancel_transfer &&
	       (file_info->file_length ? file_info->bytes_xfer < file_info->file_length : 1)) {
		readct = read(sock, (char *)&buffer, sizeof(buffer)-1);
		if (readct < 0) {
			if (debug_packets) {
				err=errno;
				fprintf(packetdump_fp, "error reading file from server: %d, %s\n", err, strerror(err));
			}
			file_info->cancel_transfer=1;
			break;
		}
		if (readct == 0) {
			if (debug_packets) {
				fprintf(packetdump_fp, "read from server, size 0\n");
			}
			break;
		}
		file_info->bytes_xfer += readct;
		fwrite(buffer, 1, readct, f);

		g_thread_yield();
	}

	fclose(f);
	close(sock);

	gdk_threads_enter();
	if (file_info->bytes_xfer == file_info->file_length) {
		snprintf(status_msg, 500, "%s. %s %s", _("File received successfully."), _("File saved as:"), file_info->fullfilename);
		file_info->status = SUCCESS;
	}
	else {
		snprintf(status_msg, 500, "%s %d, %s %d %s\n%s\n\n%s %s",
			 _("Expected"),
			 file_info->file_length,
			 _("bytes, but recieved"),
			 file_info->bytes_xfer,
			 _("bytes."),
			 _("The transfer may be incomplete due to a network error, or the sending party may have prematurely cancelled the transfer."),
			 _("File saved as:"),
			 file_info->fullfilename);
		show_ok_dialog_p(parent, status_msg );
		file_info->status = UNKNOWN;
	}

	yahoo_fxfer_filemessage(file_info->who, file_info->fullfilename, status_msg);
	gdk_flush();
	gdk_threads_leave();
	return(0);
}

/*
 * -----------------------------------------------------------------------------------------
 */

/* Creates file selector for user to enter path for saving the file
 * This is the entry path, from packet_handler 
 */
void yahoo_fxfer_show_file_selection(char *who, char *server, char *filename, char *filekey, char *token) {
	GtkWindow *parent; /* either pm of "who", or main chat window */
	GtkWidget *dialog;
	struct fxfile *file_info;
	PM_SESSION *pm_sess = find_pm_session(who);
	int rv;

	if (pm_sess) {
		parent = (GtkWindow *)pm_sess->pm_notebook->window;
	}
	else {
		parent = (GtkWindow *)chat_window;
	}

	file_info=yahoo_find_file_info(who, server, filename, filekey);
	if (!file_info) {
		/* We never found the file_info in the list.
		 * This should NEVER happen...
		 */
		return;
	}

	if (parent) {
		gtk_window_present(GTK_WINDOW(parent));
	}

	file_info->token=strdup(token);
	file_info->server=strdup(server);

	/* paranoia check */
	if (!file_download_dir) {
		file_download_dir = strdup(getenv("HOME"));
		write_config();
	}

	dialog = gtk_file_chooser_dialog_new ("Save file",
					      parent,
					      GTK_FILE_CHOOSER_ACTION_SAVE,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
					      NULL);

	gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog), TRUE);
	gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), file_download_dir);
	gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), filename);
	gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(dialog), FALSE);

	/* Create the selector */
	rv = gtk_dialog_run(GTK_DIALOG (dialog));
	if (rv == GTK_RESPONSE_ACCEPT) {
		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

		/* construct result filename */
		/* note (20/7/2008) -- I believe that the GTK_FILE_CHOOSER_ACTION_SAVE
		 * will prevent f directory from being selected. in other words, i believe
		 * that the user MUST select a path/filename combination...
		 */
		if (g_file_test(filename, G_FILE_TEST_IS_DIR)) {
			file_info->fullfilename = malloc(strlen(filename) + strlen(file_info->filename) + 2);
			strcpy(file_info->fullfilename, filename);
			strcat(file_info->fullfilename, "/");
			strcat(file_info->fullfilename, file_info->filename);
		}
		else {
			file_info->fullfilename = strdup(filename);
			free(file_info->filename);
			file_info->filename=g_path_get_basename(file_info->fullfilename);
		}

		g_free (filename);

		if (strcmp(file_download_dir, gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(dialog)))) {
			free(file_download_dir);
			file_download_dir = strdup(gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(dialog)));
			write_config();
		}

		/* fire off a thread for this file download.
		 * Each download gets a separate thread
		 */
		if (debug_packets) {
			fprintf(packetdump_fp, "Starting thread to send file: %s\n", file_info->fullfilename);
		}
		g_thread_create(yahoo_fxfer_getfile_thread, file_info, FALSE, NULL);
	}
	else {
		/* send a cancel packet */
		ymsg_fxfer_reject(ymsg_sess, file_info->who, file_info->filekey); 
		free_fxfile_info(file_info);
	}

	gtk_widget_destroy(dialog);

	return;
} /* yahoo_fxfer_show_file_selection */

void yahoo_fxfer_getfile_cancel(char *who, char *filekey) {
	struct fxfile *file_info;
	GtkWidget *parent;

	file_info=yahoo_find_file_info(who, 0, 0, filekey);
	if (!file_info) {
		/* We never found the file_info in the list.
		 * This should NEVER happen...
		 */
		return;
	}
	file_info->cancel_transfer = 1;
	parent=find_pms_window(who);
	if (!parent) parent=chat_window;
	gtk_window_present(GTK_WINDOW(parent));
	show_ok_dialog_p(parent, _("The remote user has cancelled the transfer"));
	ymsg_fxfer_cancel(ymsg_sess, who, filekey);
}

void *yahoo_fxfer_sendfile_thread(void *arg) {
	char *encoded;
	char *url;
	char *POST_strings[] = {
		"POST /relay?token=",	//  0
		0,			//  1 - encoded
		"&sender=",		//  2
		0,			//  3 - select_profile_name(YMSG_NOTIFY, 0)
		"&recver=",		//  4
		0,			//  5 - file_info->who
		" HTTP/1.1\r\n"
		"Referer: memyselfandi\r\n"
		"Cookie: ",		//  6
		0,			//  7 - ymsg_sess->cookie
		"\r\n"
		"User-Agent: " GYACH_USER_AGENT "\r\n"
		"Host: ",		//  8
		0,			//  9 - file_info->server
		"\r\n"
		"Content-Length: ",	// 10
		0,			// 11 - file_info->file_length (converted to text);
		"\r\n"
		"Cache-Control: no-cache\r\n"
		"\r\n",			// 12
		0
	};
	FILE *f;
	char buffer[512];
	char status_msg[512];
	struct sockaddr_in sa;
	int sock = -1;
	int numsnd;
	int bytesleft;
	int bytesread;
	int packets_sent;
	struct fxfile *file_info = arg;
	GtkWidget *parent; /* either pm of "who", or main chat window */
	char      *host_addr;

	file_info->bytes_xfer=0;
	/* get parent window */
	gdk_threads_enter();
	// * FIXME * get parent before thread starts.
	// * FIXME * ? maybe put it in the fxfile struct ?
	parent = find_pms_window(file_info->who);
	if (!parent) parent=chat_window;
	gtk_window_present(GTK_WINDOW(parent));
	gdk_flush();
	gdk_threads_leave();

	if (file_info->hostname) {
		sock = open_socket_for_hostname(file_info->hostname, &host_addr, parent, TRUE);
		if (sock == 0) {
			free_fxfile_info(file_info);
			return(0);
		}

		if (file_info->server) {
			free(file_info->server);
		}

		file_info->server = host_addr;
	}
	else {
		memset(&sa, 0, sizeof(sa));
		sa.sin_addr.s_addr = inet_addr(file_info->server);
		sa.sin_family = AF_INET;
		sa.sin_port = htons(80);

		if ((sock=socket(AF_INET, SOCK_STREAM, 6)) == -1) {
			gdk_threads_enter();
			show_ok_dialog_p(parent, _("Socket error!"));
			free_fxfile_info(file_info);
			gdk_flush();
			gdk_threads_leave();
			return(0);
		}

		if (connect(sock, (struct sockaddr*)&sa, sizeof(sa)) == -1) {
			gdk_threads_enter();
			show_ok_dialog_p(parent, _("Could not connect to relay server"));
			free_fxfile_info(file_info);
			gdk_flush();
			gdk_threads_leave();
			close(sock);
			return(0);
		}
	}

	if (file_info->open_callback) {
		*(char **)file_info->open_arg = strdup(file_info->server);
		/* synchronization ... */
		gdk_threads_enter();
		file_info->open_callback(file_info->open_arg);
		file_info->open_arg = NULL; /* freed by the open callback... */
		gdk_threads_leave();
		file_info->open_callback = NULL; /* only call once */
	}

	// send the "POST-command"
	encoded = url_encode(file_info->token);
	sprintf(buffer, "%i", file_info->file_length);
	POST_strings[ 1] = encoded;
	POST_strings[ 3] = select_profile_name(YMSG_NOTIFY, 0);
	POST_strings[ 5] = file_info->who;
	POST_strings[ 7] = ymsg_sess->cookie;
	POST_strings[ 9] = file_info->server;
	POST_strings[11] = buffer; // file_info->file_length (converted to text)
	url = build_string(POST_strings);
	free(encoded);
	encoded=NULL;

	write(sock, url, strlen(url));
	free(url);
	url=NULL;

	// now the data...
	if (!(f=fopen(file_info->fullfilename, "rb"))) {
		snprintf(status_msg, 500, "%s:\n%s", _("File could not be opened"), file_info->fullfilename);
		gdk_threads_enter();
		show_ok_dialog_p(parent, status_msg);
		free_fxfile_info(file_info);
		gdk_flush();
		gdk_threads_leave();
		close(sock);
		return(0);
	}

	/* create the progress meter box */
	gdk_threads_enter();
	create_meter(find_pm_session(file_info->who), file_info, Sending, file_info->show_meter);
	gdk_flush();
	gdk_threads_leave();
	file_info->status = INPROGRESS;

	packets_sent = 0;
	bytesleft = file_info->file_length;
	while (!file_info->cancel_transfer && bytesleft) {
		bytesread = fread(&buffer, 1, sizeof(buffer), f);
		numsnd = write(sock, buffer, bytesread);
		packets_sent++ ;

		if (numsnd < 0) {
			gdk_threads_enter();
			show_ok_dialog_p(parent, _("The remote user has cancelled the transfer"));
			gdk_flush();
			gdk_threads_leave();
			file_info->cancel_transfer = 1;
			break;
		}
		file_info->bytes_xfer += numsnd;
		bytesleft -= numsnd;

		g_thread_yield();
	}
	
	fclose(f);
	close(sock);

	if (file_info->cancel_transfer == 0) {
		gdk_threads_enter();
		snprintf(status_msg, 500, "%s, %s", _("File sent successfully"), file_info->fullfilename);
		yahoo_fxfer_filemessage(file_info->who, file_info->fullfilename, status_msg);
		gdk_flush();
		gdk_threads_leave();
		file_info->status = SUCCESS;
	}
	else {
		file_info->status = UNKNOWN;
	}
	return(0);
}

/*
 * -----------------------------------------------------------------------------------------
 */

/* send a file to relay-server */
/* handles our own file transfer
 * This routine gets invoked if the remote user decided to cancel the transfer.
 */
void yahoo_fxfer_sendfile_relay_cancel(char *who, char *filekey) {
	struct fxfile *file_info;
	GtkWidget *parent;

	file_info=yahoo_find_file_info(who, 0, 0, filekey);
	if (!file_info) {
		/* We never found the file_info in the list.
		 * This should NEVER happen...
		 */
		return;
	}
	file_info->cancel_transfer = 1;
	parent=find_pms_window(who);
	if (!parent) parent=chat_window;
	gtk_window_present(GTK_WINDOW(parent));
	show_ok_dialog_p(parent, _("The remote user has cancelled the transfer"));
	ymsg_fxfer_cancel(ymsg_sess, who, filekey);
}


/* send a file to relay-server */
/* handles our own file transfer
 * This is the 3rd step.
 */
void yahoo_fxfer_sendfile_relay(char *who, char *filekey, char *token) {
	struct fxfile *file_info;
		    
	file_info=yahoo_find_file_info(who, 0, 0, filekey);
	if (!file_info) {
		/* We never found the file_info in the list.
		 * This should NEVER happen...
		 */
		return;
	}

	file_info->token=strdup(token);

	/* fire off a thread for this file download.
	 * Each download gets a separate thread
	 */
	if (debug_packets) {
		fprintf(packetdump_fp, "Starting thread to receive file: %s\n", file_info->fullfilename);
	}
	g_thread_create(yahoo_fxfer_sendfile_thread, file_info, FALSE, NULL);
}


/* handles declining of file-offering */
/* Remote user has declined our file send */
void yahoo_fxfer_decline_msg(char *who, char *filekey) {
	char status_msg[512];
	GtkWidget *parent;
	struct fxfile *file_info;

	file_info=yahoo_find_file_info(who, 0, 0, filekey);
	if (file_info) {
		free_fxfile_info(file_info);
	}

	/* get parent window */
	parent = find_pms_window(who);
	if (parent) gtk_window_present(GTK_WINDOW(parent));

	snprintf(status_msg, 500, _("%s has declined the file"), who);
	show_ok_dialog_p(parent, status_msg);
}


/* preparing to send a file via relay-server */
/* Most Linux-System are firewalled and Apache listening on Port 80 */
/* So we would want to use Relay-Server when possible */
/* handles our own file transfer
 * This is the 2nd step.
 */
void yahoo_fxfer_prepsendfile(char *who, char *filekey) {
	struct fxfile *file_info;
	char *host_addr;
	int   sock;
	GtkWidget *parent;

	file_info=yahoo_find_file_info(who, 0, 0, filekey);
	if (!file_info) {
		/* We never found the file_info in the list.
		 * This should NEVER happen...
		 */
		return;
	}

	parent = find_pms_window(file_info->who);
	if (!parent) parent=chat_window;

	// get the relay-server
	sock = open_socket_for_hostname(YAHOO_RELAYSERVER, &host_addr, parent, FALSE);
	if (sock == 0) {
		return;
	}

	close(sock);

	if (file_info->server) {
		free(file_info->server);
	}

	file_info->server = host_addr;
	ymsg_fxfer_relaynotice(ymsg_sess,
			       file_info->who,
			       file_info->filename,
			       file_info->filekey,
			       file_info->server);
}


/* generates a "random" filekey */
char *yahoo_fxfer_genfilekey() {
	static char possible[] = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_$";
	char *rndcookie=(char *)malloc(25);
	int i;

	for(i=0; i<22; i++) {
		rndcookie[i] = possible[(rand() & 0x3f)];
		if (rndcookie[i] == rndcookie[i-1] && rndcookie[i] == 0x24) i--;  // avoid $$ in the first 22 bytes
	}
	rndcookie[22] = 0x24;
	rndcookie[23] = 0x24;
	rndcookie[24] = 0x00;
	return(rndcookie);
}


/* handles our own file transfer
 * This is the 1st step of the transfer.
 */
void yahoo_start_fxfer(char *who, char *filename) {
	FILE *f;
	struct fxfile *file_info;
	char status_msg[512];
	struct stat sbuf;
	GtkWidget *parent;

	/* get parent window */
	parent = find_pms_window(who);
	if (parent) gtk_window_present(GTK_WINDOW(parent));

	if (stat(filename, &sbuf)) {
		snprintf(status_msg, 500, "%s:\n%s", _("File does not exist"), filename);
		show_ok_dialog_p(parent, status_msg);
		return;
	}

	if (!(f=fopen(filename, "rb"))) {
		snprintf(status_msg, 500, "%s:\n%s", _("File could not be opened"), filename);
		show_ok_dialog_p(parent, status_msg);
		return;
	}
	fclose(f);
	
	/* Limit to files 250kb and under */
	/*	
	 * if (((int)sbuf.st_size) > 250000) {
	 *	snprintf(status_msg, 500, "%s:\n%s\n\n%s: 250kb", _("File is too large"), filename, _("Maximum Allowed File Size"));
	 *	show_ok_dialog(status_msg);
	 *	return;
	 * }
	 */

	file_info=create_fxfile_info(who, g_path_get_basename(filename), yahoo_fxfer_genfilekey(), sbuf.st_size);
	file_info->fullfilename=strdup(filename);

	ymsg_fxfer_offer(ymsg_sess, who, file_info->filename, file_info->filekey, file_info->file_length);
} /* yahoo_start_fxfer */

